#!/bin/bash

# Simple launcher script for ParVaguesViz

# Colors for terminal output
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Print banner
echo -e "${BLUE}"
echo "██████╗  █████╗ ██████╗ ██╗   ██╗ █████╗  ██████╗ ██╗   ██╗███████╗███████╗"
echo "██╔══██╗██╔══██╗██╔══██╗██║   ██║██╔══██╗██╔════╝ ██║   ██║██╔════╝██╔════╝"
echo "██████╔╝███████║██████╔╝██║   ██║███████║██║  ███╗██║   ██║█████╗  ███████╗"
echo "██╔═══╝ ██╔══██║██╔══██╗╚██╗ ██╔╝██╔══██║██║   ██║██║   ██║██╔══╝  ╚════██║"
echo "██║     ██║  ██║██║  ██║ ╚████╔╝ ██║  ██║╚██████╔╝╚██████╔╝███████╗███████║"
echo "╚═╝     ╚═╝  ╚═╝╚═╝  ╚═╝  ╚═══╝  ╚═╝  ╚═╝ ╚═════╝  ╚═════╝ ╚══════╝╚══════╝"
echo -e "                     TIDALCYCLES VISUALIZER\n${NC}"

# Get the full path to the sketch directory
SKETCH_DIR=$(cd "$(dirname "$0")" && pwd)

# Check if Processing is installed
if ! command -v processing &> /dev/null && ! command -v processing-java &> /dev/null; then
    echo -e "${RED}Error: Processing not found!${NC}"
    echo "Please make sure Processing is installed and in your PATH."
    echo "You can download it from: https://processing.org/download"
    exit 1
fi

# Check if SuperDirt might be running
if ! command -v lsof &> /dev/null || ! lsof -i :57120 &> /dev/null; then
    echo -e "${YELLOW}Warning: No process seems to be listening on port 57120.${NC}"
    echo "Make sure SuperDirt is running in SuperCollider before starting TidalCycles."
    echo -e "Typical SuperCollider startup: ${GREEN}SuperDirt.start${NC}"
    echo ""
fi

# Print troubleshooting tips
echo -e "${YELLOW}Troubleshooting tips:${NC}"
echo "1. If no visualization appears, press 'T' to generate a test pattern"
echo "2. Debug mode is ON by default, press 'D' to toggle it"
echo "3. Make sure SuperDirt is running on port 57120"

echo -e "\n${BLUE}Controls:${NC}"
echo "  D - Toggle debug info"
echo "  H - Toggle help screen"
echo "  G - Change grid style"
echo "  F - Toggle fullscreen"
echo "  R - Reset visualization"
echo "  M - Toggle metadata display"
echo "  T - Generate test pattern"
echo ""

# Set Java options to fix module restrictions (important for Processing 4.x)
export _JAVA_OPTIONS="--add-opens=java.desktop/sun.awt.X11=ALL-UNNAMED --add-opens=java.desktop/java.awt=ALL-UNNAMED -Djava.awt.headless=false -Dawt.useSystemAAFontSettings=on -Dswing.aatext=true"

# Launch Processing sketch
echo -e "${GREEN}Launching visualizer...${NC}"
echo "Sketch directory: $SKETCH_DIR"

# Try different launch methods
if command -v processing &> /dev/null; then
    # Preferred method: Use the Processing GUI
    cd "$SKETCH_DIR"
    processing "$SKETCH_DIR/ParVaguesViz.pde"
elif command -v processing-java &> /dev/null; then
    # Alternative: Use processing-java command line
    processing-java --sketch="$SKETCH_DIR" --run
else
    # Last resort: Try direct Java execution
    echo -e "${RED}Could not find Processing launcher. Trying direct execution...${NC}"
    cd "$SKETCH_DIR"
    java -jar "$SKETCH_DIR/ParVaguesViz.jar"
fi

echo -e "${GREEN}Visualizer closed.${NC}"
