/**
 * Grid class - Improved version
 * 
 * Creates a cyberpunk grid with pulse effects that react to the music
 * Reduced number of lines for cleaner visuals
 */
class Grid {
  // Grid properties
  int gridStyle = 0; // 0: standard, 1: polar, 2: hexagonal
  int numStyles = 3;
  
  // Timing properties
  float cps = 0.5;
  float pulseIntensity = 0;
  
  // Colors
  color gridColor;
  color accentColor;
  
  // Reduced line counts for cleaner visuals
  int verticalLines = 10;    // reduced from 20
  int horizontalLines = 8;   // reduced from 15
  int polarCircles = 5;     // reduced from 10
  int polarRadials = 8;     // reduced from 16
  
  // Alpha values for better visibility
  int defaultAlpha = 30;    // reduced from 50
  
  Grid() {
    gridColor = color(0, 150, 180, defaultAlpha);
    accentColor = color(0, 255, 255, 80);
  }
  
  void update() {
    // Decay pulse intensity
    pulseIntensity *= 0.95;
  }
  
  void display() {
    switch(gridStyle) {
      case 0:
        drawStandardGrid();
        break;
      case 1:
        drawPolarGrid();
        break;
      case 2:
        drawHexGrid();
        break;
    }
  }
  
  void drawStandardGrid() {
    strokeWeight(1);
    
    // Draw vertical lines (reduced count)
    float verticalSpacing = width / (float)verticalLines;
    for (int i = 0; i <= verticalLines; i++) {
      float x = i * verticalSpacing;
      float intensity = pulseIntensity * (1 - abs((x / width) - 0.5) * 2);
      
      // Only draw lines with enough visibility
      if (intensity > 0.05 || i % 2 == 0) {
        stroke(lerpColor(gridColor, accentColor, intensity));
        line(x, 0, x, height);
      }
    }
    
    // Draw horizontal lines (reduced count)
    float horizontalSpacing = height / (float)horizontalLines;
    for (int i = 0; i <= horizontalLines; i++) {
      float y = i * horizontalSpacing;
      float intensity = pulseIntensity * (1 - abs((y / height) - 0.5) * 2);
      
      // Only draw lines with enough visibility
      if (intensity > 0.05 || i % 2 == 0) {
        stroke(lerpColor(gridColor, accentColor, intensity));
        line(0, y, width, y);
      }
    }
    
    // Draw horizon line with stronger pulse
    stroke(lerpColor(gridColor, accentColor, pulseIntensity));
    strokeWeight(2 + pulseIntensity * 3);
    line(0, height * 0.5, width, height * 0.5);
  }
  
  void drawPolarGrid() {
    pushMatrix();
    translate(width / 2, height / 2);
    
    // Draw circular grid (reduced count)
    noFill();
    for (int i = 1; i <= polarCircles; i++) {
      float radius = i * (min(width, height) / (polarCircles * 2.0));
      float intensity = pulseIntensity * (1 - (i / (float)polarCircles) * 0.8);
      
      stroke(lerpColor(gridColor, accentColor, intensity));
      strokeWeight(1 + intensity * 2);
      ellipse(0, 0, radius * 2, radius * 2);
    }
    
    // Draw radial lines (reduced count)
    for (int i = 0; i < polarRadials; i++) {
      float angle = i * TWO_PI / polarRadials;
      float intensity = pulseIntensity * 0.8;
      
      stroke(lerpColor(gridColor, accentColor, intensity));
      strokeWeight(1 + intensity * 2);
      
      float radius = min(width, height) / 2;
      line(0, 0, cos(angle) * radius, sin(angle) * radius);
    }
    
    popMatrix();
  }
  
  void drawHexGrid() {
    float hexSize = 60;  // Increased from 40 for fewer hexagons
    float horizontalSpacing = hexSize * 1.5;
    float verticalSpacing = hexSize * sqrt(3);
    
    stroke(lerpColor(gridColor, accentColor, pulseIntensity * 0.5));
    strokeWeight(1 + pulseIntensity * 2);
    noFill();
    
    for (int row = -1; row < height / verticalSpacing + 1; row += 2) {  // Skip rows
      for (int col = -1; col < width / horizontalSpacing + 1; col += 2) {  // Skip columns
        float xCenter = col * horizontalSpacing + ((row % 2 == 0) ? 0 : horizontalSpacing / 2);
        float yCenter = row * verticalSpacing;
        
        // Intensity based on distance from center
        float distFromCenter = dist(xCenter, yCenter, width/2, height/2) / (width/2);
        float intensity = pulseIntensity * (1 - distFromCenter * 0.7);
        
        if (intensity > 0.05 || (row % 4 == 0 && col % 4 == 0)) {
          stroke(lerpColor(gridColor, accentColor, intensity));
          drawHexagon(xCenter, yCenter, hexSize);
        }
      }
    }
  }
  
  void drawHexagon(float xCenter, float yCenter, float size) {
    beginShape();
    for (int i = 0; i < 6; i++) {
      float angle = i * TWO_PI / 6;
      vertex(xCenter + cos(angle) * size, yCenter + sin(angle) * size);
    }
    endShape(CLOSE);
  }
  
  void trigger(float intensity) {
    // Trigger a pulse effect
    pulseIntensity = min(pulseIntensity + intensity, 1.0);
  }
  
  void setCPS(float newCps) {
    this.cps = newCps;
  }
  
  void toggleStyle() {
    gridStyle = (gridStyle + 1) % numStyles;
  }
}
